/*****************************************************************************/
/* File         regex.c                                                      */
/*                                                                           */
/* Purpose      Regular Expression Pattern Matching                          */
/*                                                                           */
/* Tested       26 APR 90       OS/2    IBM C2 1.1                           */
/*               1 MAY 90       VM      C/370                                */
/*               1 MAY 90       VM      Whitesmiths                          */
/*               7 JUL 91       AIX     RT                                   */
/*              27 MAR 91       AIX     S/6000                               */
/*                                                                           */
/* Author       Tim Bell (BELLTG at WINVMB)                                  */
/*                                                                           */
/* History                                                                   */
/*               9 MAR 89       Created                                      */
/*              20 APR 90       Added USE_SRCHSTR                            */
/*              22 OCT 90       Converted string search code to use the      */
/*                               library version of srh_search rather than   */
/*                               inline                                      */
/*              09 NOV 90       Add additional test cases and fix bug in     */
/*                               test code                                   */
/*               5 JUL 91       Fixed bug in test scripts                    */
/*               7 JUL 91       Fixed bug in handling of [^...] where the    */
/*                               matched character count was not being       */
/*                               updated                                     */
/*              18 OCT 91       Improved handling of bad program pointers    */
/*              18 OCT 91       Added support for []] matching just ]        */
/*                               and [A-] matching 'A' and '-'               */
/*              14 MAY 92       Modified syntax to avoid escaping (){}.      */
/*                              - Jeff Hamilton (JEFFH at WMAVM7)            */
/*****************************************************************************/
#define INCL_MEMORY
#define INCL_STDIO
#define INCL_STDLIB
#define INCL_STDDEF
#define INCL_LIMITS
#include "libibm.h"
#include <string.h>
#include <ctype.h>
#include "regex.h"

#if defined(DMALLOC)
#include "dmalloc.h"
#endif

/*****************************************************************************/
/* If a dump of the virtual machine as it is proceeding is required,         */
/* uncomment the following line                                              */
/*****************************************************************************/
#if 0
#define DEBUG
#endif
/*****************************************************************************/
/* If USE_SRCHSTR is TRUE, an additional section of code is used for         */
/* unanchored searching.  This uses the Boyer-Moore string search algorithm  */
/* as in srchstr.c to accelerate matching of an initial substring in the     */
/* pattern.                                                                  */
/*                                                                           */
/* This is useful if the pattern to be searched for consists of              */
/* a series of literal characters followed by a pattern.                     */
/*                                                                           */
/* There is a small overhead in both code size and pattern compile time      */
/* but this should easily be compensated for when the string to search       */
/* is non-trivial or will be matched often                                   */
/*                                                                           */
/* If the pattern does not start with literals, the normal pattern matching  */
/* techiques are used                                                        */
/*****************************************************************************/
#if !defined(USE_SRCHSTR)
#define USE_SRCHSTR 1                                      /* Default ON     */
#endif

#if defined(USE_SRCHSTR)
#include "srchstr.h"
#endif

/*****************************************************************************/
/* re_compile takes the pattern in the form of a character string and        */
/* compiles it into a program to be interpretted by a virtual machine        */
/* especially designed for matching regular expressions                      */
/* The instruction set of the virtual machine is as follows                  */
/*                                                                           */
/* ro_not_used                      Not a valid opcode                       */
/* ro_string_match <string>         Match the sequence of characters         */
/* ro_start_of_line                 Match the start of line.  If it is,      */
/*                                  continue, else raise an exception        */
/* ro_end_of_line                   Match the end of line.  If it is,        */
/*                                  continue, else raise an exception        */
/* ro_goto       [ location ]       Go to the location specified             */
/*                                  unconditionally                          */
/* ro_push_fail_vec [ location ]    If an exception is raised, go to this    */
/*                                  location.  These vectors are saved on    */
/*                                  the stack and nest                       */
/* ro_push_fail_count_vec [location] [min count] [max count ]                */
/*                                  If the count of the number of times this */
/*                                  has been executed is between min count   */
/*                                  and max count, then push an exception    */
/*                                  vector                                   */
/*                                  This is used by the \{n,m\} instruction  */
/* ro_match_any                     Match any character other than newline   */
/* ro_select <list length> { character list }                                */
/*                                  If the current character is in the       */
/*                                  character list, then continue. Else raise*/
/*                                  an exception                             */
/*                                  This is used by the [...] instruction    */
/* ro_not_select <list length> { character list }                            */
/*                                  As above, but for [^..] rather than [...]*/
/* ro_start_register [regnum]       Start saving text associated with a      */
/*                                  register                                 */
/* ro_end_register   [regnum]       Stop                                     */
/* ro_match_register [regnum]       Match the current string with the        */
/*                                  contents of register                     */
/* ro_end                           Exit                                     */
/*****************************************************************************/
char *re_error_string;                                     /* re_compile err */
#if defined(DEBUG)
void re_dump(re_prog_t *program);
#endif
#define ILLEGAL_PROGPTR ((progptr_t)(-1))

/*****************************************************************************/
/* Store the opcode specified in the program at offset progptr               */
/*****************************************************************************/
#if defined(_NO_PROTO)
static progptr_t store_opcode(program, progptr, opcode)
re_prog_t *program;
progptr_t progptr;
opcode_t opcode;
#else
static progptr_t store_opcode(re_prog_t *program,
                              progptr_t progptr,
                              opcode_t opcode)
#endif

{
  byte_t byteop=(byte_t)opcode;

  sdynarray_write(PROG_TO_CODE(program),
                  (void_ptr_t)&byteop,
                  progptr,
                  sizeof(byteop));
  return progptr+1;                                        /* Next free locn */
}

/*****************************************************************************/
/* Stores a goto at the location specified by progptr                        */
/*****************************************************************************/
#if defined(_NO_PROTO)
static progptr_t store_goto(program, progptr, opcode, locn)
re_prog_t *program;
progptr_t progptr;
opcode_t opcode;
locn_t locn;
#else
static progptr_t store_goto(re_prog_t *program,
                            progptr_t progptr,
                            opcode_t opcode,
                            locn_t locn)
#endif
{
  locn_t difflocn=locn-progptr;
  byte_t byteop=(byte_t)opcode;

  if (opcode!=ro_not_used)
  {
    sdynarray_write(PROG_TO_CODE(program),
                    (void_ptr_t)&byteop,
                    progptr,
                    sizeof(byteop));
  }
  byteop=(byte_t)((difflocn>>8)&0xFF);
  sdynarray_write(PROG_TO_CODE(program),&byteop,progptr+1,sizeof(byteop));
  byteop=(byte_t)(difflocn&0xFF);
  sdynarray_write(PROG_TO_CODE(program),&byteop,progptr+2,sizeof(byteop));
  return progptr+GOTO_SIZE;
}

/*****************************************************************************/
/* Stores a 16 bit count at the location specified by progptr                */
/* It is stored in high-low format                                           */
/*****************************************************************************/
#if defined(_NO_PROTO)
static progptr_t store_count(program, progptr, count)
re_prog_t *program;
progptr_t progptr;
unsigned count;
#else
static progptr_t store_count(re_prog_t *program,
                             progptr_t progptr,
                             unsigned count)
#endif
{
  byte_t byteop;

  byteop=(byte_t)((count>>8)&0xFF);
  sdynarray_write(PROG_TO_CODE(program),&byteop,progptr,sizeof(byteop));
  byteop=(byte_t)(count&0xFF);
  sdynarray_write(PROG_TO_CODE(program),&byteop,progptr+1,sizeof(byteop));
  return progptr+2;
}

/*****************************************************************************/
/* Read a 16 bit count at the location specified by progptr                  */
/* It is stored in high-low format                                           */
/*****************************************************************************/
#if defined(_NO_PROTO)
static unsigned read_count(program, progptr)
re_prog_t *program;
progptr_t progptr;
#else
static unsigned read_count(re_prog_t *program,
                           progptr_t progptr)
#endif
{
  unsigned count;

  count=SDYNARRAY_QUICK_READ(PROG_TO_CODE(program),progptr,byte_t);
  count<<=8;
  count+=SDYNARRAY_QUICK_READ(PROG_TO_CODE(program),progptr+1,byte_t);
  return count;
}

/*****************************************************************************/
/* Read the location to jump to from a program.  The locations are stored    */
/* relative to the current program counter so that they are easier to        */
/* relocate.                                                                 */
/*****************************************************************************/
#if defined(_NO_PROTO)
static progptr_t read_locn(program, progptr)
re_prog_t *program;
progptr_t progptr;
#else
static progptr_t read_locn(re_prog_t *program,
                           progptr_t progptr)
#endif
{
  int16_t locn;

  locn=SDYNARRAY_QUICK_READ(PROG_TO_CODE(program),progptr,byte_t);
  locn<<=8;
  locn+=SDYNARRAY_QUICK_READ(PROG_TO_CODE(program),progptr+1,byte_t);
  return (progptr_t)locn+progptr-1;
}

/*****************************************************************************/
/* Insert a goto at location insptr in the program.  It should relocate      */
/* from insptr to progptr and also update p1, p2, and p3 so that they        */
/* point to the correct addresses                                            */
/*****************************************************************************/
#if defined(_NO_PROTO)
static progptr_t insert_goto(program, progptr, opcode, insptr, locn,p1,p2,p3)
re_prog_t *program;
progptr_t progptr;
opcode_t opcode;
progptr_t insptr;
locn_t locn;
progptr_t *p1;
progptr_t *p2;
progptr_t *p3;
#else
static progptr_t insert_goto(re_prog_t *program,
                             progptr_t progptr,
                             opcode_t opcode,
                             progptr_t insptr,
                             locn_t locn,
                             progptr_t *p1,
                             progptr_t *p2,
                             progptr_t *p3)
#endif
{
  int i=0;
  byte_t temp;

  sdynarray_write(PROG_TO_CODE(program),&i,progptr+3,GOTO_SIZE);

  for (i=progptr;i>=insptr;--i)
  {
    temp=SDYNARRAY_QUICK_READ(PROG_TO_CODE(program),i,byte_t);
    sdynarray_write(PROG_TO_CODE(program),&temp,i+GOTO_SIZE,sizeof(temp));
  }
  (void)store_goto(program,insptr,opcode,locn);
  if (p1!=NULL && *p1!=ILLEGAL_PROGPTR && *p1<=progptr && *p1>=insptr)
    *p1+=GOTO_SIZE;
  if (p2!=NULL && *p2!=ILLEGAL_PROGPTR && *p2<=progptr && *p2>=insptr)
    *p2+=GOTO_SIZE;
  if (p3!=NULL && *p3!=ILLEGAL_PROGPTR && *p3<=progptr && *p3>=insptr)
    *p3+=GOTO_SIZE;
  return progptr+GOTO_SIZE;
}

/*****************************************************************************/
/* Insert a goto count at location insptr in the program.  It should relocate*/
/* from insptr to progptr and also update p1, p2, and p3 so that they        */
/* point to the correct addresses                                            */
/*****************************************************************************/
#if defined(_NO_PROTO)
static progptr_t insert_goto_count(
  program, progptr, opcode, insptr, locn,p1,p2,p3)
re_prog_t *program;
progptr_t progptr;
opcode_t opcode;
progptr_t insptr;
locn_t locn;
progptr_t *p1;
progptr_t *p2;
progptr_t *p3;
#else
static progptr_t insert_goto_count(re_prog_t *program,
                                   progptr_t progptr,
                                   opcode_t opcode,
                                   progptr_t insptr,
                                   locn_t locn,
                                   progptr_t *p1,
                                   progptr_t *p2,
                                   progptr_t *p3)
#endif
{
  int i=0;
  byte_t temp;

  sdynarray_write(PROG_TO_CODE(program),
                  (void_ptr_t)&i,
                  progptr+3,
                  GOTO_SIZE);

  for (i=progptr;i>=insptr;--i)
  {
    temp=SDYNARRAY_QUICK_READ(PROG_TO_CODE(program),i,byte_t);
    sdynarray_write(
      PROG_TO_CODE(program),
      &temp,
      i+GOTO_COUNT_SIZE,
      sizeof(temp));
  }
  (void)store_goto(program,insptr,opcode,locn);
  if (p1!=NULL && *p1!=ILLEGAL_PROGPTR && *p1<=progptr && *p1>=insptr)
    *p1+=GOTO_COUNT_SIZE;
  if (p2!=NULL && *p2!=ILLEGAL_PROGPTR && *p2<=progptr && *p2>=insptr)
    *p2+=GOTO_COUNT_SIZE;
  if (p3!=NULL && *p3!=ILLEGAL_PROGPTR && *p3<=progptr && *p3>=insptr)
    *p3+=GOTO_COUNT_SIZE;
  return progptr+GOTO_COUNT_SIZE;
}

/*****************************************************************************/
/* Store a start_register, stop_register or match_register instruction       */
/*****************************************************************************/
#if defined(_NO_PROTO)
static progptr_t store_register(program, progptr, opcode, regnum)
re_prog_t *program;
progptr_t progptr;
opcode_t opcode;
unsigned regnum;
#else
static progptr_t store_register(re_prog_t *program,
                                progptr_t progptr,
                                opcode_t opcode,
                                unsigned regnum)
#endif
{
  byte_t regbyte=(byte_t)regnum;

  (void)store_opcode(program,progptr,opcode);
  sdynarray_write(PROG_TO_CODE(program),&regbyte,progptr+1,sizeof(regbyte));
  return progptr+2;
}

/*****************************************************************************/
/* Start a character list instruction. beg_select is set to point to the     */
/* byte count that should be incremented each time a character is added      */
/*****************************************************************************/
#if defined(_NO_PROTO)
static progptr_t store_start_char_list(program, progptr, opcode, beg_select)
re_prog_t *program;
progptr_t progptr;
opcode_t opcode;
progptr_t *beg_select;
#else
static progptr_t store_start_char_list(re_prog_t *program,
                                       progptr_t progptr,
                                       opcode_t opcode,
                                       progptr_t *beg_select)
#endif
{
  byte_t count=0;

  progptr=store_opcode(program,progptr,opcode);
  sdynarray_write(PROG_TO_CODE(program),&count,progptr,sizeof(count));
  if (beg_select!=NULL)
    *beg_select=progptr;
  return progptr+1;
}

/*****************************************************************************/
/* Add a character to a previous list instruction. beg_select points to the  */
/* byte count that should be incremented each time a character is added      */
/*****************************************************************************/
#if defined(_NO_PROTO)
static progptr_t store_char_list(program, progptr, beg_list, selection)
re_prog_t *program;
progptr_t progptr;
progptr_t beg_list;
char selection;
#else
static progptr_t store_char_list(re_prog_t *program,
                                 progptr_t progptr,
                                 progptr_t beg_list,
                                 char selection)
#endif
{
  byte_t count;

  count=SDYNARRAY_QUICK_READ(PROG_TO_CODE(program),beg_list,byte_t);
  SDYNARRAY_QUICK_WRITE(PROG_TO_CODE(program),count+1,beg_list,byte_t);
  sdynarray_write(PROG_TO_CODE(program),&selection,progptr,sizeof(selection));
  return progptr+1;
}

/*****************************************************************************/
/* Compile a regular expression.                                             */
/*****************************************************************************/
#if defined(_NO_PROTO)
re_prog_t *re_compile(patptr)
char *patptr;
#else
re_prog_t *re_compile(char *patptr)
#endif
{
  progptr_t progptr=0;                                     /* Program counter*/
  char c;                                                  /* Pattern char   */
  bool_t error=FALSE;                                      /* Found an error */
  char d;

  /***************************************************************************/
  /* The address of a goto in the code previously generated that needs       */
  /* to be patched up                                                        */
  /***************************************************************************/
  progptr_t patch_goto=ILLEGAL_PROGPTR;

  /***************************************************************************/
  /* The address of the last completed expression                            */
  /***************************************************************************/
  progptr_t lastexprptr=ILLEGAL_PROGPTR;

  /***************************************************************************/
  /* The address of the last enclosing bracket or the start of the           */
  /* regexpr                                                                 */
  /***************************************************************************/
  progptr_t lastbracket=0;

  dynarray_t stack;                               /* stack for parsing       */
  parse_stack_t stack_frame;                      /* used to read/write stack*/
  re_prog_t *program;
  int stackptr=0;                                 /* Stack empty             */
  opcode_t opcode;
  progptr_t beg_select;
  unsigned register_count=1;                      /* Use reg 1 first         */
  long count;
  char *count_end;
  byte_t register_match;
  char *end_of_register_text;
  progptr_t laststringmatch=ILLEGAL_PROGPTR;

  /***************************************************************************/
  /*  Create the stack.  This will extend as required                        */
  /***************************************************************************/
  if (dynarray_create(&stack,sizeof(parse_stack_t))==NULL)
    return NULL;

  /***************************************************************************/
  /* Program space is allocated from the heap.  We need to                   */
  /* 1) Claim memory for all of the pointers and counters                    */
  /* 2) Claim memory for the actual program itself                           */
  /***************************************************************************/
  program=(re_prog_t *)malloc(sizeof(re_prog_t));
  if (program==NULL)
  {
    dynarray_destroy(&stack);
    return NULL;                                           /* Error          */
  }
  if (sdynarray_create(PROG_TO_CODE(program))==NULL)
  {
    dynarray_destroy(&stack);
    free((void_ptr_t)program);
    return NULL;
  }
  program->start_of_line_only=FALSE;                       /* Assume no ^    */
  program->initialised_table=FALSE;
  program->table=NULL;

  while (*patptr!='\0' && error==FALSE)
  {
    c= *patptr++;                                          /* Read pattern   */
#if defined(DEBUG)
    (void)store_opcode(program,progptr,ro_end);
    printf("Handling %c progptr=%d",c,progptr);
    re_dump(program);
#endif
    switch (c)
    {
    case '$':                                              /* End of line    */
      if (*patptr=='\0')
      {
        progptr=store_opcode(program,progptr,ro_end_of_line);
        laststringmatch=ILLEGAL_PROGPTR;                   /*End lit match   */
      }
      else
        goto char_match;
      break;

    case '^':                                              /* Start of line  */
      if (progptr==0)
      {
        progptr=store_opcode(program,progptr,ro_start_of_line);
        lastbracket=progptr;
        program->start_of_line_only=TRUE;
        laststringmatch=ILLEGAL_PROGPTR;                   /*End lit match   */
      }
      else
        goto char_match;
      break;

    case '*':
      /***********************************************************************/
      /* Match previous expression multiple times                            */
      /* This should generate the following code                             */
      /* Assuming the last expression generated <code>, the new code         */
      /* be                                                                  */
      /*        ..........................                                   */
      /*        .                        V                                   */
      /* push_fail_vec <code> goto                                           */
      /*          A             .                                            */
      /*          ...............                                            */
      /***********************************************************************/
      if (lastexprptr==ILLEGAL_PROGPTR)
      {
        re_error_string="Must preceed * with a regular expression";
        error=TRUE;
      }
      else
      {
        laststringmatch=ILLEGAL_PROGPTR;                   /*End lit match   */
        progptr=insert_goto(
          program,
          progptr,
          ro_push_fail_vec,
          lastexprptr,
          progptr+2*GOTO_SIZE,
          &patch_goto,
          &lastexprptr,
          &lastbracket);
        progptr=store_goto(program,progptr,ro_goto,lastexprptr-GOTO_SIZE);
      }
      break;

    case '.':                                              /* Match Any      */
      lastexprptr = progptr;
      progptr=store_opcode(program,progptr,ro_match_any);
      laststringmatch=ILLEGAL_PROGPTR;                     /* End lit match  */
      break;

    case '[':                                              /* Match selection*/
      laststringmatch=ILLEGAL_PROGPTR;                     /* End lit match  */
      opcode=ro_select;
      lastexprptr=progptr;                                 /* For [c]*       */
      switch(*patptr)
      {
      case '^':                                            /* Not selection  */
        opcode=ro_not_select;
        ++patptr;
        break;
      default:
        break;
      }
      c= *patptr++;
      progptr=store_start_char_list(program,progptr,opcode,&beg_select);
      do
      {
        if (c=='\0') {
          re_error_string="End of string encountered inside [";
          error=TRUE;
        } else {
          /* Characters preceded by backslash have special meaning */
          if (c == '\\') {
             c = *patptr++;
             switch (c) {
             case 'n':
                c = '\n';
                break;
             case 'r':
                c = '\r';
                break;
             case 't':
                c = '\t';
                break;
             case 'f':
                c = '\f';
                break;
             default:
                /* Keep the quoted value */
                break;
             }
          }
          progptr=store_char_list(program,progptr,beg_select,c);
          if (*patptr=='-')                                /* Handle range  */
          {
            /*****************************************************************/
            /* The default, if there is no range is a range of c-c           */
            /*****************************************************************/
            d=(char)(c+1);
            ++patptr;                                      /* Skip '-'      */
            c= *patptr++;
            if (c==']')
            {
              patptr-=2;                                   /* Step  to '-' */
            }
            else if (c=='\0')
            {
              re_error_string="Range selection without final range";
              error=TRUE;
            }
            else if (c<d)
            {
              re_error_string=
                "First character in range selection is less than second";
              error=TRUE;
            }
            else
            {
              while (d<=c)
              {
                progptr=store_char_list(program,progptr,beg_select,d);
                ++d;
              }
            }
          }
          if (error==FALSE)
          {
            c= *patptr++;
          }
        }
      } while (c!=']' && error==FALSE);
      break;

    case '|':
      /***********************************************************************/
      /* Match alternative expressions                                       */
      /* This should generate the following code                             */
      /* Assuming the last expression generated a|b,    the new code         */
      /* be                                                                  */
      /*        ....................................                         */
      /*        .                                  V                         */
      /* push_fail_vec 'a' goto                   'b'                        */
      /*                     .                                    A          */
      /*                     ......................................          */
      /***********************************************************************/
      laststringmatch=ILLEGAL_PROGPTR;                     /*End lit match   */
      if (lastexprptr==ILLEGAL_PROGPTR)
      {
        re_error_string="Must preceed | with a regular expression";
        error=TRUE;
      }
      else
      {
        progptr=insert_goto (
          program,
          progptr,
          ro_push_fail_vec,
          lastbracket,
          progptr+2*GOTO_SIZE,
          &patch_goto,
          &lastexprptr,
          &lastbracket);
        if (patch_goto!=ILLEGAL_PROGPTR)
          (void)store_goto(program,patch_goto, ro_not_used, progptr);
        patch_goto=progptr;
        progptr=store_goto(program,progptr,ro_goto,0);
        lastbracket=progptr;
        lastexprptr=ILLEGAL_PROGPTR;
      }
      break;

    case '+':
    case '?':
      /***********************************************************************/
      /* Match previous expression one or more times                         */
      /* This is equivalent to \{1,\}                                        */
      /***********************************************************************/
      if (lastexprptr==ILLEGAL_PROGPTR)
      {
        re_error_string=(c=='+')?"Must preceed + with a regular expression":
                                 "Must preceed ? with a regular expression";
        error=TRUE;
      }
      else
      {
        laststringmatch=ILLEGAL_PROGPTR;                     /*End lit match*/
        progptr=insert_goto_count(
          program,
          progptr,
          ro_push_fail_count_vec,
          lastexprptr,
          progptr+GOTO_SIZE+GOTO_COUNT_SIZE,
          &patch_goto,
          &lastexprptr,
          &lastbracket);
        (void)store_count(program,lastexprptr-4,(unsigned)((c=='+')?1:0));
#if defined(CC_AIXPS2)
        (void)store_count(program,lastexprptr-2,(unsigned)((c=='+')?65535:1));
#else
        (void)store_count(program,lastexprptr-2,(unsigned)((c=='+')?65535u:1));
#endif
        progptr=store_goto(
          program,
          progptr,
          ro_goto,
          lastexprptr-GOTO_COUNT_SIZE);
#if 0
        /* Having one or more does make the expression invalid */
        lastexprptr=ILLEGAL_PROGPTR;
#endif
      }
      break;
   case '(':                                            /*  ( re  )       */
     stack_frame.lastexprptr=progptr;                   /*Make stack frame*/
     stack_frame.lastbracket=lastbracket;
     stack_frame.patch_goto=patch_goto;
     if (register_count<RE_MAX_REGISTERS-1)             /*Set up \n value */
     {
       progptr=store_register(
         program,
         progptr,
         ro_start_register,
         register_count);
       stack_frame.register_count=register_count;
     }
     else
     {
       stack_frame.register_count=RE_MAX_REGISTERS;
     }
     /*********************************************************************/
     /* Next  (  ) will have a new register                               */
     /*********************************************************************/
     ++register_count;
     dynarray_write(&stack,&stack_frame,stackptr);      /*Push onto stack */
     lastexprptr=ILLEGAL_PROGPTR;                       /*No last express */
     lastbracket=progptr;                               /*Save last ( pos */
     laststringmatch=ILLEGAL_PROGPTR;                   /*End lit match   */
     ++stackptr;
     break;
   case ')':
     if (stackptr==0)                                   /* Previous (     */
     {
       re_error_string="Unmatched )";
       error=TRUE;
     }
     else
     {
       --stackptr;                                      /* Pop frame      */
       dynarray_read(&stack,&stack_frame,stackptr);
       lastbracket=stack_frame.lastbracket;             /* Restore state  */
       lastexprptr=stack_frame.lastexprptr;
       /*******************************************************************/
       /*  If there are any gotos which need to be patched up, then       */
       /*  we had better patch them up here as the patch_goto variable    */
       /*  is about to get updated.                                       */
       /*******************************************************************/
       if (patch_goto!=ILLEGAL_PROGPTR)
         (void)store_goto(program,patch_goto,ro_not_used,progptr);
      /********************************************************************/
      /* Register recording needs to be completed, store an end register  */
      /* opcode to complete it.                                           */
      /********************************************************************/
       if (stack_frame.register_count!=RE_MAX_REGISTERS)
       {
         progptr=store_register(
           program,
           progptr,
           ro_end_register,
           stack_frame.register_count);
       }
       patch_goto=stack_frame.patch_goto;
       laststringmatch=ILLEGAL_PROGPTR;                 /*End lit match   */
     }
     break;
   case '{':
     /*********************************************************************/
     /* Match previous expression multiple times                          */
     /* This should generate the following code                           */
     /* Assuming the last expression generated <code>, the new code       */
     /* be                                                                */
     /*        ..........................                                 */
     /*        .                        V                                 */
     /* push_fail_count_vec <code> goto                                   */
     /*          A                  .                                     */
     /*          ....................                                     */
     /*                                                                   */
     /* Push fail count vec will only push an exception handler if        */
     /* the number of times it has been executed exceeds the lower        */
     /* limit of the count.                                               */
     /* Thus, if the expression does not match enough times, it will      */
     /* not recurse.                                                      */
     /*********************************************************************/
     if (lastexprptr==ILLEGAL_PROGPTR)
     {
       re_error_string="Must preceed { with a regular expression";
       error=TRUE;
     }
     else
     {
       laststringmatch=ILLEGAL_PROGPTR;                 /*End lit match   */
       progptr=insert_goto_count(
         program,
         progptr,
         ro_push_fail_count_vec,
         lastexprptr,
         progptr+GOTO_SIZE+GOTO_COUNT_SIZE,
         &patch_goto,
         &lastexprptr,
         &lastbracket);
       if (*patptr==',')                                  /*  { ,         */
       {
         count=0;
         count_end=patptr;
       }
       else if (isdigit((int)*patptr))                    /*  { n         */
       {
         count=strtol(patptr,&count_end,0);               /* Read count   */
       }
       else
       {
         re_error_string="Number or , expected after {";
         error=TRUE;
       }
     }

     if (count_end==NULL && error==FALSE)               /* Read number    */
     {
       re_error_string="End of string not expected in {";
       error=TRUE;
     }

     if (error==FALSE)
     {
       (void)store_count(program,lastexprptr-4,(unsigned)count);
                                                        /* Update program */
       if (*count_end==',')                             /*  { n ,         */
       {
         ++count_end;                                   /* Skip comma     */
         if (!isdigit((int)*count_end))
         {
#if defined(CC_AIXPS2)
           count=65535;                                 /*  { n ,  }      */
#else
           count=65535u;                                /*  { n ,  }      */
#endif
           patptr=count_end+1;
         }
         else
         {
           count=strtol(count_end,&patptr,0);           /*  { n , m       */
           if (patptr==NULL)                            /* No number ?    */
           {
             re_error_string="End of string not expected in {";
             error=TRUE;
           }
         }
       }
       else if ((*count_end=='\\' && count_end[1]=='}') || *count_end=='}')
       {
         patptr=count_end;                              /* { n }        */
       }
       else
       {
         re_error_string="Number or , expected after {<number>,";
         error=TRUE;
       }
     }

     if (error==FALSE)
     {
       if (*patptr=='\\')                               /* Read \         */
         ++patptr;
       if (*patptr=='}')                                /* Read }         */
       {
         (void)store_count(program,lastexprptr-2,(unsigned)count);
         progptr=store_goto(
           program,
           progptr,
           ro_goto,
           lastexprptr-GOTO_COUNT_SIZE);
         ++patptr;
       }
       else
       {
         re_error_string="Expecting } after {[number],[number";
         error=TRUE;
       }
     }
     break;

    /*************************************************************************/
    /* Escaped characters                                                    */
    /*************************************************************************/
    case '\\':
      c= *patptr++;                                        /* Char after \   */
      switch(c)
      {
      case '\0':
        re_error_string=
          "End of string encountered immediately after escape character (\\)";
        error=TRUE;
        break;
      case '\n':
        re_error_string=
          "Return encountered immediately after escape character (\\)";
        error=TRUE;
        break;
      case 'n':                                            /* \n is newline  */
        c='\n';
        goto char_match;
      case 'r':                                            /* \r is return  */
        c='\r';
        goto char_match;
      case 'f':                                            /* \f is form feed */
        c='\f';
        goto char_match;
      case 't':                                            /* \t is TAB      */
        c='\t';
        goto char_match;
      case '0':
      case '1':
      case '2':
      case '3':
      case '4':
      case '5':
      case '7':
      case '8':
      case '9':
        /*********************************************************************/
        /* Handle \n register match                                          */
        /*********************************************************************/
        register_match=(byte_t)strtol(patptr-1,&end_of_register_text,0);
        laststringmatch=ILLEGAL_PROGPTR;                   /*End lit match   */
        if (end_of_register_text==NULL ||
            register_match==0 ||
            register_match >= RE_MAX_REGISTERS)
        {
          re_error_string="Invalid number for register \\<number>";
          error=TRUE;
        }
        else
        {
          progptr=store_register(
            program,
            progptr,
            ro_match_register,
            register_match);
        }
        break;
      default:
        goto char_match;
      }
      break;

    default:                                               /* Match character*/
    char_match:
      /***********************************************************************/
      /* ab* has to match 'a' followed by zero or more bs.  Thus, we need to */
      /* break any active string match so that the 'a' is matched as a       */
      /* separate expression compared to the b*                              */
      /***********************************************************************/
      switch(*patptr)
      {
      case '\\':
        if (patptr[1]!='{')
          break;
      case '*':
      case '+':
        laststringmatch=ILLEGAL_PROGPTR;
        break;
      default:
        break;
      }

      if (laststringmatch==ILLEGAL_PROGPTR)
      {
        lastexprptr=progptr;                               /* For c*         */
        progptr=store_start_char_list(
          program,
          progptr,
          ro_string_match,
          &laststringmatch);
      }
      progptr=store_char_list(program,progptr,laststringmatch,c);
      break;
    }
  }

  /***************************************************************************/
  /* If there is an outstanding goto to be patched in, this is the locn. to  */
  /* jump to.                                                                */
  /***************************************************************************/
  if (patch_goto!=ILLEGAL_PROGPTR)
    (void)store_goto(program,patch_goto, ro_not_used, progptr);
  progptr=store_opcode(program,progptr,ro_end);

  if (stackptr != 0)                                       /* Still got (    */
  {
     if (error == FALSE) {
        re_error_string="Unmatched ( at end of regular expression";
        error=TRUE;
     }
  }

  dynarray_destroy(&stack);                                /* Free resources */
  if (error)
  {
    re_destroy(program);                                   /* Error, no prog */
    program=NULL;
  }

  return program;
}

#if defined(DEBUG)

#if defined(_NO_PROTO)
void re_dump(program)
re_prog_t *program;
#else
void re_dump(re_prog_t *program)
#endif
{
  int progptr=0;
  bool_t finished=FALSE;
  int inc=0;
  int count;
  opcode_t opcode;

  while (finished==FALSE)
  {
    printf("\n%2d: ",progptr);
    inc=1;
    opcode=
      (opcode_t)(SDYNARRAY_QUICK_READ(PROG_TO_CODE(program),progptr,byte_t));
    switch(opcode)
    {
    case ro_end:
      printf("[End]");
      finished=TRUE;
      break;

    case ro_match_any:
      printf("[Any]");
      break;

    case ro_start_of_line:
      printf ("[Start of Line]");
      break;

    case ro_end_of_line:
      printf("[End of Line]");
      break;

    case ro_push_fail_vec:
      printf("[Push Fail Vec] %u",read_locn(program,progptr+1));
      inc=GOTO_SIZE;
      break;

    case ro_goto:
      printf("[Goto] %u",read_locn(program,progptr+1));
      inc=GOTO_SIZE;
      break;

    case ro_string_match:
      printf ("[String Match] \"");
      count=(int)SDYNARRAY_QUICK_READ(PROG_TO_CODE(program),progptr+1,byte_t);
      inc=2;
      while (count>0)
      {
        printf(
          "%c",
          SDYNARRAY_QUICK_READ(PROG_TO_CODE(program),progptr+inc,byte_t));
        --count;
        ++inc;
      }
      printf("\"");
      break;

    case ro_select:
    case ro_not_select:
      if (opcode==ro_select)
      {
        printf("[Select] \"");
      }
      else
      {
        printf("[!Select] \"");
      }
      count=(int)SDYNARRAY_QUICK_READ(PROG_TO_CODE(program),progptr+1,byte_t);
      inc=2;
      while (count>0)
      {
        printf(
          "%c",
          SDYNARRAY_QUICK_READ(PROG_TO_CODE(program),progptr+inc,byte_t));
        --count;
        ++inc;
      }
      printf("\"");
      break;

    case ro_start_register:
      printf(
        "[Start Register] %d",
        SDYNARRAY_QUICK_READ(PROG_TO_CODE(program),progptr+1,byte_t));
      inc=2;
      break;

    case ro_end_register:
      printf(
        "[End Register] %d",
        SDYNARRAY_QUICK_READ(PROG_TO_CODE(program),progptr+1,byte_t));
      inc=2;
      break;

    case ro_match_register:
      printf("[Match Register] %d",SDYNARRAY_QUICK_READ(PROG_TO_CODE(program),progptr+1,byte_t));
      inc=2;
      break;

    case ro_push_fail_count_vec:
      printf(
        "[Push Fail Count Vec] %u %u %u",
        read_locn(program,progptr+1),
        read_count(program,progptr+3),
        read_count(program,progptr+5));
      inc=GOTO_COUNT_SIZE;
      break;

    default:
      printf("[Unknown]");
      printf("Unknown opcode %x\n",opcode);
      exit(1);
    }
    progptr+=inc;
  }
  printf("\n");
}
#endif

/*****************************************************************************/
/* Procedure    stub_re_search                                               */
/*                                                                           */
/* Purpose      Searches a string for a regular expression, using the stack  */
/*              supplied.                                                    */
/*                                                                           */
/* Parameters                                                                */
/*              program         The compiled version of the regular          */
/*                              expression                                   */
/*                                                                           */
/*              string          The string to try and match to               */
/*                                                                           */
/*              stack           The space to store run time local variables  */
/*                              and status information.                      */
/*                                                                           */
/* Return Code                                                               */
/*              >=0             The number of characters matched by the      */
/*                              pattern.                                     */
/*                                                                           */
/*              -1              The pattern did not match the string at all. */
/*****************************************************************************/
#if defined(_NO_PROTO)
static int sub_re_search (program, string, stack)
re_prog_t *program;
char *string;
dynarray_t *stack;
#else
static int sub_re_search (re_prog_t *program,
                          char *string,
                          dynarray_t *stack)
#endif
{
  progptr_t progptr=0;
  int       stackptr=0;
  bool_t    finished=FALSE;
  bool_t    matched;
  int       stringptr=0;
#if defined(DEBUG)
  char     *opstring;
  int       stringorig;
  int       progorig;
#endif
  progptr_t selectptr;
  int       selectcount;
  char     *select_listptr;
  run_stack_t stack_frame;
  opcode_t    opcode;
  unsigned  regnum;
  unsigned max_regnum=0;
  unsigned count=0;
  progptr_t count_location=ILLEGAL_PROGPTR;
  unsigned start;                                          /* For registers  */
  unsigned end;                                            /* For registers  */
  char *string_match;                                      /* Match literals */

  while(finished==FALSE)
  {
    matched=TRUE;
#if defined(DEBUG)
    stringorig=stringptr;
    progorig=progptr;
    printf("String: %s\n",&string[stringptr]);
#endif
    ++progptr;
    opcode=
      (opcode_t)(SDYNARRAY_QUICK_READ(PROG_TO_CODE(program),progptr-1,byte_t));
    switch(opcode)
    {
    case ro_end:
#if defined(DEBUG)                                         /* End of program */
      opstring="[End]";
#endif
      matched=TRUE;
      finished=TRUE;
      break;

      /***********************************************************************/
      /* '.' matches any character except for end of line and end of string  */
      /***********************************************************************/
    case ro_match_any:                                     /* .              */
#if defined(DEBUG)
      opstring="[Any]";
#endif
      switch(string[stringptr])
      {
      case '\n':                                           /* !match newline */
      case '\0':                                           /* !match EOS     */
        matched=FALSE;
        break;
      default:
        ++stringptr;                                       /* Matched        */
        break;
      }
      break;

      /***********************************************************************/
      /* '^' matches start of a line or start of string                      */
      /***********************************************************************/
    case ro_start_of_line:                                 /* ^              */
#if defined(DEBUG)
      opstring="[Start of Line]";
#endif
      if (stringptr!=0 && string[stringptr-1]!='\n')       /* Start of string*/
                                                           /* or start line  */
        matched=FALSE;
      break;

      /***********************************************************************/
      /* [...] matches any character stored within the range                 */
      /***********************************************************************/
    case ro_select:                                        /* Match [...]    */
#if defined(DEBUG)
      opstring="[Select]";
#endif
      selectptr=progptr+1;                                 /* Start of chars */
      select_listptr=(char *)SDYNARRAY_ADDRESS(PROG_TO_CODE(program),progptr);
      selectcount=(byte_t)*select_listptr++;
      progptr+=selectcount+1;
      matched=FALSE;
      while (selectcount>0)                                /* List all done ?*/
      {
        if (string[stringptr]==*select_listptr)
        {
          ++stringptr;                                     /* Exact match    */
          matched=TRUE;
          break;
        }
        else if (string[stringptr]=='\0')                  /* End of string  */
          break;
        --selectcount;                                     /* One less char  */
        ++select_listptr;
      }
      break;

    case ro_not_select:                                    /* [^...]         */
#if defined(DEBUG)
      opstring="[!Select]";
#endif
      selectptr=progptr+1;
      select_listptr=(char *)SDYNARRAY_ADDRESS(PROG_TO_CODE(program),progptr);
      selectcount=(byte_t)*select_listptr++;
      progptr+=selectcount+1;
      matched=TRUE;
      while (selectcount>0)
      {
        if (string[stringptr]==*select_listptr)
        {
          matched=FALSE;
          ++stringptr;
          break;
        }
        else if (string[stringptr]=='\0')
        {
          matched=FALSE;
          break;
        }
        ++select_listptr;
        --selectcount;
      }
      /***********************************************************************/
      /* If we've matched, skip over the matched character                   */
      /***********************************************************************/
      if (matched)
        ++stringptr;
      break;

    case ro_end_of_line:                                   /* $              */
#if defined(DEBUG)
      opstring="[End of Line]";
#endif
      if (string[stringptr]=='\n')                         /* End of line    */
        ++stringptr;
      else if (string[stringptr]=='\0')                    /* End of string  */
        ;
      else
        matched=FALSE;                                     /* Not end of line*/
      break;

    case ro_push_fail_vec:
#if defined(DEBUG)
      opstring="[Push Fail Vec]";
#endif
      stack_frame.progptr=progptr-1;                       /*location to jump*/
      stack_frame.string=stringptr;                        /* Retry from here*/
      stack_frame.destination=read_locn(program,progptr);
      stack_frame.count=count;                             /* How many \{ \} */
      stack_frame.count_location=count_location;
      dynarray_write(stack,&stack_frame,stackptr);
      progptr+=GOTO_SIZE-1;                                /*Skip address    */
      ++stackptr;                                          /*Push stack      */
      break;

    case ro_push_fail_count_vec:                           /*Used by \{n,n\} */
#if defined(DEBUG)
      opstring="[Push Fail Count Vec]";
#endif
      if (count_location!=progptr)
      {
        count_location=progptr;
        count=0;                                           /*Nested \{ \}    */
      }
      if (count>=read_count(program,progptr+4))            /*count >= m      */
      {
        progptr=read_locn(program,progptr);
      }
      else if (count>=read_count(program,progptr+2))       /*count >= n      */
      {
        stack_frame.progptr=progptr-1;
        stack_frame.string=stringptr;
        stack_frame.destination=read_locn(program,progptr);
        stack_frame.count=count;
        stack_frame.count_location=count_location;
        dynarray_write(stack,&stack_frame,stackptr);
        ++stackptr;
        ++count;
        progptr+=GOTO_COUNT_SIZE-1;
      }
      else
      {
        ++count;                                           /*count < n       */
        progptr+=GOTO_COUNT_SIZE-1;
      }
      break;

    case ro_goto:
#if defined(DEBUG)
      opstring="[Goto]";
#endif
      progptr=read_locn(program,progptr);                  /*Jump uncond.    */
      break;

    case ro_string_match:
#if defined(DEBUG)
      opstring="[String Match]";
#endif
     selectptr=progptr+1;                                  /*Start of chars  */
     string_match=(char *)SDYNARRAY_ADDRESS(PROG_TO_CODE(program),selectptr);
     selectcount=SDYNARRAY_QUICK_READ(PROG_TO_CODE(program),progptr,byte_t);
     progptr+=selectcount+1;
     matched=TRUE;
     while (selectcount-->0)                               /*List all done ? */
     {
       if (string[stringptr++]!= *string_match++)
       {
         matched=FALSE;
         break;
       }
     }
     break;

    case ro_start_register:                                /*Start of \(     */
#if defined(DEBUG)
      opstring="[Start Register]";
#endif
      regnum=SDYNARRAY_QUICK_READ(PROG_TO_CODE(program),progptr,byte_t);
      program->active[regnum]=TRUE;                        /*Register on     */
      program->start[regnum]=stringptr;                    /*Start of reg    */
      program->end[regnum]=0;                              /*No end yet      */
      ++progptr;
      if (regnum>max_regnum)
        max_regnum=regnum;
      break;

    case ro_end_register:
#if defined(DEBUG)
      opstring="[End Register]";
#endif
      regnum=SDYNARRAY_QUICK_READ(PROG_TO_CODE(program),progptr,byte_t);
      program->active[regnum]=TRUE;                        /*Register still  */
      program->end[regnum]=stringptr;                      /*End of register */
      ++progptr;
      break;

    case ro_match_register:
#if defined(DEBUG)
      opstring="[Match Register]";
#endif
      regnum=SDYNARRAY_QUICK_READ(PROG_TO_CODE(program),progptr,byte_t);
      if (regnum>=RE_MAX_REGISTERS || program->active[regnum]==FALSE)
        ;                                                  /*Matched         */
      else
      {
        start=program->start[regnum];                      /*Read register   */
        end=program->end[regnum];
        if (start>=end)                                    /*Nowt to compare */
          ;                                                /*Matched         */
        else
        {
          if (strncmp(&string[stringptr],
                      &program->string[start],
                      (size_t)(end-start))!=0)
            matched=FALSE;
          else
            stringptr+=end-start;                          /*Matched         */
        }
      }
      ++progptr;
      break;
    default:
    case ro_not_used:
#if defined(DEBUG)
      opstring="[Unknown]";
#endif
      printf("Unknown opcode %x\n",opcode);
      exit(1);
    }

#if defined(DEBUG)
   printf("%s %s\n",opstring,(matched == FALSE) ? "FALSE" : "TRUE");
#endif

    if (matched==FALSE)                                    /*Failed to match */
    {
      if (stackptr!=0)                                     /*Back track      */
      {
        --stackptr;                                        /*Resume last push*/
        dynarray_read(stack,&stack_frame,stackptr);
        progptr=stack_frame.destination;
        count=stack_frame.count;
        count_location=stack_frame.count_location;
        stringptr=stack_frame.string;
      }
      else
      {
        /*********************************************************************/
        /* There is no more fail vectors on the stack, so we have failed to  */
        /* match.  Return an error                                           */
        /*********************************************************************/
        finished=TRUE;
      }
    }
  }
#if defined(DEBUG)
  printf("Does %smatch\n",(matched == FALSE) ? "not " : "");
#endif
  if (matched==TRUE)
    return stringptr;                                      /*Num chars match */

  do
  {
    program->active[max_regnum]=FALSE;
  } while (max_regnum--);
  return -1;                                               /* Error          */
}

/*****************************************************************************/
/* Procedure    re_search                                                    */
/*                                                                           */
/* Purpose      Main entry point for regular expression searching after      */
/*              a pattern has been compiled using re_compile.                */
/*                                                                           */
/* Parameters                                                                */
/*              program         The compiled version of the regular          */
/*                              expression                                   */
/*                                                                           */
/*              string          The string to try and match to               */
/*                                                                           */
/* Return Code                                                               */
/*              >=0             The number of characters matched by the      */
/*                              pattern.                                     */
/*                                                                           */
/*              -1              The pattern did not match the string at all. */
/*****************************************************************************/
#if defined(_NO_PROTO)
int re_search (program, string)
re_prog_t *program;
char *string;
#else
int re_search (re_prog_t *program, char *string)
#endif
{
  dynarray_t stack;
  int rc;
  int i;

  /***************************************************************************/
  /* stub_re_search does most of the work, but needs to have a stack created */
  /* for it                                                                  */
  /***************************************************************************/
  if (program==NULL)
    return -1;
  for (i=0;i<RE_MAX_REGISTERS;++i)                         /*Clear registers */
    program->active[i]=FALSE;
  if (dynarray_create(&stack,sizeof(run_stack_t))==NULL)   /*Run time stack  */
    return -1;
  program->string=string;                                  /*Needed for \n   */
  rc=sub_re_search(program,string,&stack);
  dynarray_destroy(&stack);                                /*Run over        */
  return rc;
}

#if USE_SRCHSTR
/*****************************************************************************/
/* Procedure    re_initialise_table                                          */
/*                                                                           */
/* Purpose      Creates the rapid search table that can speed up fixed       */
/*              string pattern matching at the beginning of a pattern        */
/*                                                                           */
/* Parameters                                                                */
/*              program         The compiled regular expression              */
/*                                                                           */
/*              searchfor       The fixed match string                       */
/*                                                                           */
/*              length          The length of the sub-string                 */
/*****************************************************************************/
#if defined(_NO_PROTO)
static void re_initialise_table(program, searchfor, length)
re_prog_t *program;
unsigned char *searchfor;
int length;
#else
static void re_initialise_table(re_prog_t *program,
                                unsigned char *searchfor,
                                int length)
#endif
{
  if (program->initialised_table==FALSE)
  {
    program->table=(void_ptr_t)srh_compile(searchfor, length);
    program->initialised_table=TRUE;
  }
  return;
}

/*****************************************************************************/
/* Procedure    re_skip_early_exact                                          */
/*                                                                           */
/* Purpose      Skips to a point where a string could match                  */
/*              This will only be successful if there is an exact string     */
/*              match at the beginning of the regexp                         */
/*                                                                           */
/* Parameters                                                                */
/*              program         The compiled regular expression              */
/*                                                                           */
/*              string          The match string                             */
/*                                                                           */
/* Return Code  The pointer to where to start matching the string to         */
/*****************************************************************************/
#if defined(_NO_PROTO)
static char *re_skip_early_exact(program, string)
re_prog_t *program;
char *string;
#else
static char *re_skip_early_exact(re_prog_t *program, char *string)
#endif
{
  int offset;
  size_t string_length;

  if (program->initialised_table==FALSE ||                 /* No table avail */
    program->table==NULL)
    return string;

  string_length=strlen(string);

  offset=srh_search((srh_prog_t *)program->table, (byte_t *)string, string_length);
  if (offset==-1)
    return NULL;
  return &string[offset];
}
#endif

/*****************************************************************************/
/* Procedure    re_search_unanchored                                         */
/*                                                                           */
/* Purpose      Main entry point for unanchored pattern searching after      */
/*              a pattern has been compiled using re_compile.                */
/*                                                                           */
/* Parameters                                                                */
/*              program         The compiled version of the regular          */
/*                              expression                                   */
/*                                                                           */
/*              string          The string to try and match to               */
/*                                                                           */
/*              offset          The offset that the pattern matched.         */
/*                                                                           */
/* Return Code                                                               */
/*              >=0             The number of characters matched by the      */
/*                              pattern.                                     */
/*                                                                           */
/*              -1              The pattern did not match the string at all. */
/*****************************************************************************/
#if defined(_NO_PROTO)
int re_search_unanchored(program, string, offset)
re_prog_t *program;
char *string;
int *offset;
#else
int re_search_unanchored(re_prog_t *program, char *string, int *offset)
#endif
{
  char *try_string=string;
  int length;                                              /* Length of match*/
  dynarray_t stack;                                        /* Run time stack */
  int i;

  if (program==NULL)
    return -1;

  for (i=0;i<RE_MAX_REGISTERS;++i)                         /* Clear registers*/
    program->active[i]=FALSE;

  /***************************************************************************/
  /* By allocating the stack early, we can avoid having to reallocate it     */
  /* every time we call re_search                                            */
  /***************************************************************************/
  if (dynarray_create(&stack,sizeof(run_stack_t))==NULL)   /* Run time stack */
    return -1;
  /***************************************************************************/
  /* If the pattern starts with a ^, then we can skip the whole line if      */
  /* the pattern fails to match.                                             */
  /***************************************************************************/
  if (program->start_of_line_only)                         /* ^pattern       */
  {
    do
    {
      program->string=try_string;                          /* Needed for \n  */
      length=sub_re_search(program,try_string,&stack);
      if (length!= -1)                                     /* Matched ?      */
      {
        if (offset)
          *offset = (int) (try_string-string);
        dynarray_destroy(&stack);                          /* Run over       */
        return length;                                     /* Length of match*/
      }
      try_string=strchr(try_string,'\n');                  /*Goto newline    */
      if (try_string==NULL)                                /* Found ?        */
        break;
    } while (*++try_string!=EOS);
  }
  else
  {
#if USE_SRCHSTR
    /*************************************************************************/
    /* Does the program start with the following code                        */
    /*                                                                       */
    /*  ro_string_match "string" ....                                        */
    /*                                                                       */
    /* If so, we can do a faster string match on the start of the program    */
    /* using the Boyer-Moore fast string match algorithm.                    */
    /*************************************************************************/
    if (SDYNARRAY_QUICK_READ(PROG_TO_CODE(program),0,byte_t)==
      (byte_t)ro_string_match)
    {
      /***********************************************************************/
      /* Length of initial text string                                       */
      /***********************************************************************/
      length=(int)SDYNARRAY_QUICK_READ(PROG_TO_CODE(program),1,byte_t);
      /***********************************************************************/
      /* Construct an initial map of characters                              */
      /***********************************************************************/
      re_initialise_table(
        program,
        (unsigned char *)SDYNARRAY_ADDRESS(PROG_TO_CODE(program),2),
        length);
    }
    else
      program->initialised_table=FALSE;                    /* No table used  */
#else
    program->initialised_table=FALSE;                      /* No table used  */
#endif

    do
    {
#if USE_SRCHSTR
      /***********************************************************************/
      /* If there is an table to do a fast string search on, skip to the     */
      /* first likely match point                                            */
      /***********************************************************************/
      if (program->initialised_table)
      {
        try_string=re_skip_early_exact(program,try_string);
        if (try_string==NULL)
          break;
      }
#endif

      /***********************************************************************/
      /* See if the string matched the regular expression                    */
      /***********************************************************************/
      program->string=try_string;
      length=sub_re_search(program,try_string,&stack);
      if (length!= -1)                                     /* Matched ?      */
      {
        if (offset)                                        /* Match offset   */
          *offset = (int) (try_string-string);
        dynarray_destroy(&stack);
        return length;                                     /* Length of match*/
      }
    } while (*++try_string!=EOS);
  }
  /***************************************************************************/
  /* No match found                                                          */
  /***************************************************************************/
  if (offset!=NULL)                                        /* Invalid offset */
    *offset= -1;
  dynarray_destroy(&stack);
  return -1;                                               /* No match       */
}

/*****************************************************************************/
/* Procedure    re_destroy                                                   */
/*                                                                           */
/* Purpose      Free up the memory associated with a given pattern program   */
/*                                                                           */
/* Parameters                                                                */
/*              program         The compiled version of the regular          */
/*                              expression                                   */
/*****************************************************************************/
#if defined(_NO_PROTO)
void re_destroy(program)
re_prog_t *program;
#else
void re_destroy(re_prog_t *program)
#endif
{
  if (program==NULL)
    return;

  sdynarray_destroy(PROG_TO_CODE(program));                /* Kill program   */

#if USE_SRCHSTR
  if (program->initialised_table && program->table!=NULL)
    srh_destroy((srh_prog_t *)program->table);
#endif

  free(program);
}

/*****************************************************************************/
/* Procedure    re_read_register                                             */
/*                                                                           */
/* Purpose      Reads the contents of a register after a pattern has matched */
/*                                                                           */
/* Parameters                                                                */
/*              program         The compiled version of the regular          */
/*                              expression                                   */
/*                                                                           */
/*              return_string   The string to store the contents of the      */
/*                              register in.  If this is NULL, some memory   */
/*                              is allocated off the heap.                   */
/*                                                                           */
/*              string_length   The length of 'return_string'.  If return    */
/*                              string is NULL, this parameter is ignored.   */
/*                                                                           */
/*              regnum          The register number to read the value from   */
/*                                                                           */
/* Return Code  A pointer to the return string (or the malloc'd area)        */
/*                                                                           */
/*              NULL indicates out of memory or illegal register value       */
/*****************************************************************************/
#if defined(_NO_PROTO)
char *re_read_register(program, return_string, string_length, regnum)
re_prog_t *program;
char *return_string;
unsigned string_length;
unsigned regnum;
#else
char *re_read_register(re_prog_t *program,
                       char *return_string,
                       unsigned string_length,
                       unsigned regnum)
#endif
{
  unsigned start;
  unsigned end;

  if (program==NULL)
    return NULL;

  /***************************************************************************/
  /* Check for illegal register values                                       */
  /***************************************************************************/
  if (regnum>=RE_MAX_REGISTERS || program->active[regnum]==FALSE)
    return NULL;

  /***************************************************************************/
  /* Valid register contents that will fit in to the string provided ?       */
  /***************************************************************************/
  start=program->start[regnum];
  end=program->end[regnum];

  if (start>end)                                           /* Incomplete     */
    return NULL;

  if (end-start+1>string_length && return_string!=NULL)
    return NULL;

  if (return_string==NULL)                                 /* Off the heap ? */
    return_string=(char *)malloc(end-start+1);

  if (return_string==NULL)
    return NULL;

  memcpy(return_string,program->string+start,(size_t)(end-start));
  return_string[end-start]=EOS;
  return return_string;
}

#if defined(TEST)
/*****************************************************************************/
/* Set of test cases                                                         */
/*****************************************************************************/
static struct
{
  char *pat;                                               /* Pattern        */
  char *match;                                             /* String to match*/
  int anchored_result;                                     /* Anchored search*/
  int unanchored_result;                                   /* Unanchored     */
  int unanchored_offset;                                   /* Offset of match*/
} test_array[]=
{
  { "a","a",1,1,0 },
  { "a","ba",-1,1,1 },
  { "a","ab",1,1,0 },
  { "a","b",-1,-1,-1 },
  { "^a","a",1,1,0 },
  { "^a","b",-1,-1,-1 },
  { "^a","ba",-1,-1,-1 },
  { "^a","ab",1,1,0 },
  { "^a$","a",1,1,0 },
  { "^a$","ab",-1,-1,-1 },
  { "[a-c]","a",1,1,0 },
  { "[a-c]","b",1,1,0 },
  { "[a-c]","c",1,1,0 },
  { "[a-c]","d",-1,-1,-1 },
  { "[-abc]","a",1,1,0 },
  { "[-abc]","-",1,1,0 },
  { "[-abc]","d",-1,-1,-1 },
  { "[^abc]","a",-1,-1,-1 },
  { "[^abc]","b",-1,-1,-1 },
  { "[^abc]","c",-1,-1,-1 },
  { "[^abc]","d",1,1,0 },
  { "[^-abc]", "a", -1, -1, -1 },
  { "[^-abc]", "-", -1, -1, -1 },
  { "[^-abc]", "d", 1, 1, 0 },
  { "[]^-]*", "-]^", 3, 3, 0 },
  { "[^]]*", "ABC]DEF", 3, 3, 0 },
  { "a[a-c]*","ab",2,2,0},
  { "a[a-ce-i]*","abfffffffffffij",14,14,0},
  { "^ab*$","abbb",4,4,0},
  { "^ab*$","abab",-1,-1,-1},
  { "a*","aaaa",4,4,0 },
  { "a*","dddd",0,0,0 },
  { "a*","",0,0,0 },
  { "a*b","aaaab",5,5,0 },
  { "a*b","aaaa",-1,-1,-1 },
  { "a|b","a",1,1,0 },
  { "a|b","b",1,1,0 },
  { "a|b","c",-1,-1,-1 },
  { "\\(a|b\\)","a",1,1,0 },
  { "\\(a|b\\)","b",1,1,0 },
  { "\\(a|b\\)","c",-1,-1,-1 },
  { "\\(a|b\\)a\\(a|b\\)","aac",-1,-1,-1 },
  { "\\(a|b\\)a\\(a|b\\)","aaa",3,3,0 },
  { "(a|b)","c",-1,-1,-1 },
  { "(a|b)","(a",2,2,0 },
  { "^\\(a|b\\)*$","abbbaa",6,6,0 },
  { "^\\(a|b\\)*$","b",1,1,0 },
  { "^\\(a|b\\)*$","c",-1,-1,-1 },
  { "a*","aaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaa\
aaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaa",140,140,0 },
  { "aa*","aaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaa\
aaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaa",140,140,0 },
  { "aaaaaaaaaaaa*","aaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaa\
aaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaa",140,140,0 },
  { "aaaabaaaaaaa*","aaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaa\
aaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaa",-1,-1,-1 },
  { "a*b","aaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaa\
aaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaac",-1,-1,-1 },
  { "^a\\{1,2\\}a$","a",-1,-1,-1 },
  { "^a\\{1,2\\}a$","aa",2,2,0 },
  { "^a\\{1,2\\}a$","aaa",3,3,0 },
  { "^a\\{1,2\\}a$","aaaa",-1,-1,-1 },
  { "^a\\{1,2\\}a$","aaaaa",-1,-1,-1 },
  { "^a\\{1\\}a$","aa",2,2,0 },
  { "^a\\{1\\}a$","a",-1,-1,-1 },
  { "^a\\{1\\}a$","aaa",-1,-1,-1 },
  { "^a\\{2,\\}a$","aaa",3,3,0 },
  { "^a\\{2,\\}a$","aa",-1,-1,-1 },
  { "^a\\{0,1\\}b$","b",1,1,0},
  { "^a\\{0,1\\}b$","ab",2,2,0},
  { "^a\\{0,1\\}b$","c",-1,-1,-1},
  { "a\\{2,\\}","aaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaa\
aaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaa",140,140,0 },
  { "^a?b$","b",1,1,0},
  { "^a?b$","ab",2,2,0},
  { "^a?ab$","aab",3,3,0},
  { "^a?ab$","ab",2,2,0},
  { "^a?b$","c",-1,-1,-1},
  { "^\\(a*\\)b\\1$","aaabaaa",7,7,0 },
  { "^\\(a*\\)b\\1$","aaabaaaa",-1,-1,-1 },
  { "^\\(a*\\)b\\1$","aaabaa",-1,-1,-1 },
  { "^\\(a*\\)b\\1$","b",1,1,0 },
  { "\\(.*\\)b\\(.*\\)b\\1\\2$","aaabccbaacc",-1,10,1 },
  { "\\(.*\\)b\\(.*\\)b\\1\\2$","bccbcc",6,6,0 },
  { "^\\(.*\\)b\\(.*\\)b\\1\\2$","aabccbcc",-1,-1,-1 },
  { "^\\(.*\\)b\\(.*\\)b\\1\\2","aabccbaaccdd",10,10,0 },
  { "^\\ta\\t\\nb$","\ta\t\nb\n",6,6,0 },
  { "^\\ta\\t\\nb$","\ta\t\nb",5,5,0 },
  { "^\\ta\\t\\nb$","\ta\tb",-1,-1,-1 },
  { "\\(a*\\(b*\\)\\)","aaaabbbbbb",10,10,0 },
  { "^test","this is a\ntest",-1,4,10 },
  { "test\ntest$","this is a test\ntest",-1,9,10 },
  { "a+","notsucceed",-1,-1,-1},
  { "a+","a",1,1,0 },
  { "a+","aa",2,2,0 },
  { "a+","baa",-1,2,1 }
};

#if defined(_NO_PROTO)
int main()
#else
int main(void)
#endif
{
  char pat[80];
  char expr[80];
  re_prog_t *program;
  int i;
  int anchored_match;
  int unanchored_match;
  int unanchored_offset;
  char *reg;
  FILE *fp;
  char *file_contents;
  long file_length;
  long file_start;
  int rc=0;
  unsigned regnum;

  fprintf(stderr,"Regular Expression tester\n");
  fprintf(stderr,"A pattern of 'test' will run the standard ship test\n");
  fprintf(stderr,"An expression of '<filename' will use the file filename as input\n");
  do
  {
    fprintf(stderr,"Enter pattern:");
    *pat=EOS;
    if (gets(pat)==NULL)
      break;
    if (*pat)
    {
      if (strcmp(pat,"test")==0)
      {
        for (i=0;i<COUNTOF(test_array);++i)
        {
          program=re_compile(test_array[i].pat);
          anchored_match=re_search(program,test_array[i].match);
          unanchored_match=re_search_unanchored(program,test_array[i].match,&unanchored_offset);
          if (anchored_match!=test_array[i].anchored_result ||
              unanchored_match!=test_array[i].unanchored_result ||
              unanchored_offset!=test_array[i].unanchored_offset)
          {
            printf("%s %s\n",test_array[i].pat,test_array[i].match);
            if (anchored_match!=test_array[i].anchored_result)
              printf("  Failed anchored match got %d expected %d\n",anchored_match,test_array[i].anchored_result);
            if (unanchored_match!=test_array[i].unanchored_result)
              printf("  Failed unanchored match got %d expected %d\n",unanchored_match,test_array[i].unanchored_result);
            if (unanchored_offset!=test_array[i].unanchored_offset)
              printf("  Failed unanchored offset got %d expected %d\n",unanchored_offset,test_array[i].unanchored_offset);
            rc=1;
          }
          re_destroy(program);
        }
      }
      else if (strcmp(pat,"quit")==0)
      {
        *pat=0;
      }
      else
      {
        if ((program=re_compile(pat))==NULL)
        {
          printf("Error: %s\n",re_error_string);
          rc=1;
        }
        else
        {
#if defined(DEBUG)
          re_dump(program);
#endif
          fprintf(stderr,"Enter expression to match to %s:",pat);
          *expr=EOS;
          if (gets(expr)==NULL)
            break;
          /*******************************************************************/
          /* If the expression is "<filename" then we read from the file and */
          /* do an unanchored search on it.  This is similar to the Unix     */
          /* grep utility.                                                   */
          /*******************************************************************/
          if (*expr=='<')
          {
            fp=fopen(expr+1,"r");
            if (fp==NULL)
            {
              printf("Error: cannot open %s for reading\n",expr+1);
              perror("");
            }
            else
            {
              file_start=ftell(fp);
              fseek(fp,(long)0,SEEK_END);
              file_length=ftell(fp);
              fseek(fp,file_start,SEEK_SET);
              file_contents=(char *)malloc((size_t)file_length);
              if (file_contents!=NULL)
              {
                fread(file_contents,(size_t)file_length,1,fp);
                fclose(fp);
                unanchored_match=
                  re_search_unanchored(program,
                                       file_contents,
                                       &unanchored_offset);
                if (unanchored_match== -1)
                {
                  printf("Error: Not found\n");
                  rc=1;
                }
                else
                {
                  fprintf(stderr,"%d %d [%.10s]\n",unanchored_match,unanchored_offset,&file_contents[unanchored_offset]);
                }
                free(file_contents);
              }
            }
          }
          else
          {
            fprintf(stderr, "rc=%d\n",re_search(program,expr));
          }
          for (regnum=0;regnum<RE_MAX_REGISTERS;++regnum)
          {
            reg=re_read_register(program,(char *)NULL,0,regnum);
            if (reg!=NULL)
            {
              fprintf(stderr,"%u. %s\n",regnum,reg);
              free(reg);
            }
          }
          re_destroy(program);
        }
      }
    }
  } while (*pat);
  return rc;
}
#endif

